# ------------------------------------------------------------------------------
# AWS Provider Configuration
# ------------------------------------------------------------------------------
provider "aws" {
  region = var.aws_region
}

# ------------------------------------------------------------------------------
# Data Sources
# ------------------------------------------------------------------------------
data "aws_availability_zones" "available" {}

data "aws_route53_zone" "primary" {
  name         = trimsuffix(var.domain_name, ".")
  private_zone = false
}

# Get the latest Ubuntu AMI
data "aws_ami" "ubuntu" {
  most_recent = true
  owners      = ["099720109477"] # Canonical (Ubuntu) owner ID

  filter {
    name   = "name"
    values = ["ubuntu/images/hvm-ssd-gp3/ubuntu-noble-24.04-amd64-server-*"]
  }

  filter {
    name   = "virtualization-type"
    values = ["hvm"]
  }
}

# ------------------------------------------------------------------------------
# Locals
# ------------------------------------------------------------------------------
locals {
  app_port            = 8181
  instance_subdomains = [for i in range(var.instance_num): format("%s-%d", var.common_subdomain, i + 1)]
  setup_script_key    = "setup/${var.setup_archive_name}"
}

# ------------------------------------------------------------------------------
# S3 Bucket for Setup Files
# ------------------------------------------------------------------------------
resource "aws_s3_bucket" "setup_files" {
  bucket_prefix = "${var.project_name}-setup-"
  force_destroy = true
  tags = {
    Name = "${var.project_name}-setup-bucket"
  }
}

resource "aws_s3_bucket_public_access_block" "setup_files" {
  bucket = aws_s3_bucket.setup_files.id

  block_public_acls       = true
  block_public_policy     = true
  ignore_public_acls      = true
  restrict_public_buckets = true
}

resource "aws_s3_object" "setup_archive" {
  bucket = aws_s3_bucket.setup_files.id
  key    = local.setup_script_key
  source = var.setup_archive_path
  etag   = filemd5(var.setup_archive_path)
  tags = {
    Name = "Setup Archive"
  }
}

# ------------------------------------------------------------------------------
# EFS File System (Shared storage for all EC2 instances)
# ------------------------------------------------------------------------------
resource "aws_efs_file_system" "shared" {
  creation_token = "${var.project_name}-efs"
  encrypted      = true

  tags = {
    Name = "${var.project_name}-efs"
  }
}

resource "aws_efs_mount_target" "shared" {
  count           = length(aws_subnet.public)
  file_system_id  = aws_efs_file_system.shared.id
  subnet_id       = aws_subnet.public[count.index].id
  security_groups = [aws_security_group.efs_sg.id]
}

resource "aws_security_group" "efs_sg" {
  name        = "${var.project_name}-efs-sg"
  description = "Allow NFS traffic from EC2 instances"
  vpc_id      = aws_vpc.main.id

  ingress {
    from_port       = 2049
    to_port         = 2049
    protocol        = "tcp"
    security_groups = [aws_security_group.ec2_sg.id]
    description     = "Allow NFS from EC2"
  }

  egress {
    from_port   = 0
    to_port     = 0
    protocol    = "-1"
    cidr_blocks = ["0.0.0.0/0"]
    description = "Allow all outbound"
  }

  tags = {
    Name = "${var.project_name}-efs-sg"
  }
}

# ------------------------------------------------------------------------------
# VPC Configuration
# ------------------------------------------------------------------------------
resource "aws_vpc" "main" {
  cidr_block           = var.vpc_cidr
  enable_dns_support   = true
  enable_dns_hostnames = true
  tags = {
    Name = "${var.project_name}-vpc"
  }
}

resource "aws_internet_gateway" "gw" {
  vpc_id = aws_vpc.main.id
  tags = {
    Name = "${var.project_name}-igw"
  }
}

resource "aws_subnet" "public" {
  count                   = length(var.public_subnet_cidrs)
  vpc_id                  = aws_vpc.main.id
  cidr_block              = var.public_subnet_cidrs[count.index]
  availability_zone       = data.aws_availability_zones.available.names[count.index % length(data.aws_availability_zones.available.names)]
  map_public_ip_on_launch = true
  tags = {
    Name = "${var.project_name}-public-subnet-${count.index + 1}"
  }
}

resource "aws_subnet" "private" {
  count                   = length(var.private_subnet_cidrs)
  vpc_id                  = aws_vpc.main.id
  cidr_block              = var.private_subnet_cidrs[count.index]
  availability_zone       = data.aws_availability_zones.available.names[count.index % length(data.aws_availability_zones.available.names)]
  map_public_ip_on_launch = false
  tags = {
    Name = "${var.project_name}-private-subnet-${count.index + 1}"
  }
}

resource "aws_route_table" "public" {
  vpc_id = aws_vpc.main.id
  route {
    cidr_block = "0.0.0.0/0"
    gateway_id = aws_internet_gateway.gw.id
  }
  tags = {
    Name = "${var.project_name}-public-rt"
  }
}

resource "aws_route_table_association" "public" {
  count          = length(aws_subnet.public)
  subnet_id      = aws_subnet.public[count.index].id
  route_table_id = aws_route_table.public.id
}

resource "aws_route_table" "private_rds" {
  vpc_id = aws_vpc.main.id
  tags = {
    Name = "${var.project_name}-private-rds-rt"
  }
}

resource "aws_route_table_association" "private_rds" {
  count          = length(aws_subnet.private)
  subnet_id      = aws_subnet.private[count.index].id
  route_table_id = aws_route_table.private_rds.id
}

# ------------------------------------------------------------------------------
# Security Groups
# ------------------------------------------------------------------------------
resource "aws_security_group" "alb_sg" {
  name        = "${var.project_name}-app-alb-sg"
  description = "Allow HTTPS (443) inbound traffic to App ALBs"
  vpc_id      = aws_vpc.main.id

  ingress {
    from_port   = 443
    to_port     = 443
    protocol    = "tcp"
    cidr_blocks = ["0.0.0.0/0"]
  }
  egress {
    from_port   = 0
    to_port     = 0
    protocol    = "-1"
    cidr_blocks = ["0.0.0.0/0"]
  }
  tags = { Name = "${var.project_name}-app-alb-sg" }
}

resource "aws_security_group" "ec2_sg" {
  name        = "${var.project_name}-app-ec2-sg"
  description = "Allow App traffic (8181) from App ALB only. Public IP exists for outbound internet access, but inbound is restricted to ALB."
  vpc_id      = aws_vpc.main.id

  ingress {
    from_port       = local.app_port
    to_port         = local.app_port
    protocol        = "tcp"
    security_groups = [aws_security_group.alb_sg.id]
    description     = "Allow traffic from ALB only"
  }
  egress {
    from_port   = 0
    to_port     = 0
    protocol    = "-1"
    cidr_blocks = ["0.0.0.0/0"]
    description = "Allow all outbound traffic"
  }
  tags = { Name = "${var.project_name}-app-ec2-sg" }
}

resource "aws_security_group" "rds_sg" {
  name        = "${var.project_name}-rds-sg"
  description = "Allow MySQL traffic from App EC2 instances"
  vpc_id      = aws_vpc.main.id
  ingress {
    from_port       = 3306
    to_port         = 3306
    protocol        = "tcp"
    security_groups = [aws_security_group.ec2_sg.id]
  }
  egress {
    from_port   = 0
    to_port     = 0
    protocol    = "-1"
    cidr_blocks = ["0.0.0.0/0"]
  }
  tags = { Name = "${var.project_name}-rds-sg" }
}

# ------------------------------------------------------------------------------
# Application Load Balancer
# ------------------------------------------------------------------------------
resource "aws_lb" "app_alb" {
  name               = "${var.project_name}-alb"
  internal           = false
  load_balancer_type = "application"
  security_groups    = [aws_security_group.alb_sg.id]
  subnets            = [for subnet in aws_subnet.public : subnet.id]
  enable_deletion_protection = false
  tags = {
    Name = "${var.project_name}-alb"
  }
}

# Main target group for all instances (sync.domain)
resource "aws_lb_target_group" "app_tg_main" {
  name        = "${var.project_name}-tg-main"
  port        = local.app_port
  protocol    = "HTTP"
  vpc_id      = aws_vpc.main.id
  target_type = "instance"

  health_check {
    enabled             = true
    interval            = 30
    path                = var.health_check_path
    port                = "traffic-port"
    protocol            = "HTTP"
    timeout             = 5
    healthy_threshold   = 2
    unhealthy_threshold = 2
  }
  stickiness {
    type            = "lb_cookie"
    cookie_duration = 3600
    enabled         = true
  }
  tags = {
    Name = "${var.project_name}-tg-main"
  }
}

# Individual target groups for each instance (sync-1.domain, sync-2.domain, etc.)
resource "aws_lb_target_group" "app_tg_individual" {
  count       = var.instance_num
  name        = "${var.project_name}-tg-${count.index + 1}"
  port        = local.app_port
  protocol    = "HTTP"
  vpc_id      = aws_vpc.main.id
  target_type = "instance"

  health_check {
    enabled             = true
    interval            = 30
    path                = var.health_check_path
    port                = "traffic-port"
    protocol            = "HTTP"
    timeout             = 5
    healthy_threshold   = 2
    unhealthy_threshold = 2
  }
  tags = {
    Name = "${var.project_name}-tg-${count.index + 1}"
  }
}

# HTTPS Listener with default action
resource "aws_lb_listener" "https_listener" {
  load_balancer_arn = aws_lb.app_alb.arn
  port              = 443
  protocol          = "HTTPS"
  ssl_policy        = "ELBSecurityPolicy-2016-08"
  certificate_arn   = var.acm_certificate_arn

  default_action {
    type             = "forward"
    target_group_arn = aws_lb_target_group.app_tg_main.arn
  }
}

# Listener rules for individual instances (sync-1.domain, sync-2.domain, etc.)
resource "aws_lb_listener_rule" "instance_routing" {
  count        = var.instance_num
  listener_arn = aws_lb_listener.https_listener.arn
  priority     = 100 + count.index

  action {
    type             = "forward"
    target_group_arn = aws_lb_target_group.app_tg_individual[count.index].arn
  }

  condition {
    host_header {
      values = ["${local.instance_subdomains[count.index]}.${var.domain_name}"]
    }
  }
}

# ------------------------------------------------------------------------------
# IAM Role for EC2
# ------------------------------------------------------------------------------
resource "aws_iam_role" "ec2_role" {
  name = "${var.project_name}-ec2-role"

  assume_role_policy = jsonencode({
    Version = "2012-10-17"
    Statement = [
      {
        Action = "sts:AssumeRole"
        Effect = "Allow"
        Principal = {
          Service = "ec2.amazonaws.com"
        }
      }
    ]
  })

  tags = {
    Name = "${var.project_name}-ec2-role"
  }
}

resource "aws_iam_role_policy_attachment" "ec2_ssm_policy" {
  role       = aws_iam_role.ec2_role.name
  policy_arn = "arn:aws:iam::aws:policy/AmazonSSMManagedInstanceCore"
}

resource "aws_iam_role_policy" "ec2_s3_access" {
  name = "${var.project_name}-ec2-s3-access"
  role = aws_iam_role.ec2_role.id

  policy = jsonencode({
    Version = "2012-10-17"
    Statement = [
      {
        Effect = "Allow"
        Action = [
          "s3:GetObject",
          "s3:ListBucket"
        ]
        Resource = [
          aws_s3_bucket.setup_files.arn,
          "${aws_s3_bucket.setup_files.arn}/*"
        ]
      }
    ]
  })
}

resource "aws_iam_instance_profile" "ec2_profile" {
  name = "${var.project_name}-ec2-profile"
  role = aws_iam_role.ec2_role.name

  tags = {
    Name = "${var.project_name}-ec2-profile"
  }
}

# ------------------------------------------------------------------------------
# EC2 Instances
# ------------------------------------------------------------------------------
resource "aws_instance" "app_ec2" {
  count                  = var.instance_num
  ami                    = var.app_ami_id != null ? var.app_ami_id : data.aws_ami.ubuntu.id
  instance_type          = var.ec2_instance_type
  key_name               = var.ec2_key_name
  iam_instance_profile   = aws_iam_instance_profile.ec2_profile.name
  vpc_security_group_ids = [aws_security_group.ec2_sg.id]
  subnet_id              = aws_subnet.public[count.index % length(aws_subnet.public)].id
  
  user_data = <<-EOF
              #!/bin/bash
              set -e
              
              # Log output
              exec > >(tee /var/log/user-data.log)
              exec 2>&1
              
              echo "=== Starting CData Sync setup ==="
              echo "Timestamp: $(date)"
              
              # Install required packages
              echo "Installing required packages..."
              apt-get update -y
              apt-get install -y \
                nfs-common \
                unzip \
                curl
              
              # Install AWS CLI v2 (recommended for Ubuntu)
              echo "Installing AWS CLI v2..."
              if ! command -v aws &> /dev/null; then
                cd /tmp
                curl "https://awscli.amazonaws.com/awscli-exe-linux-x86_64.zip" -o "awscliv2.zip"
                unzip -q awscliv2.zip
                ./aws/install
                rm -rf awscliv2.zip aws
                echo "AWS CLI installed: $(aws --version)"
              else
                echo "AWS CLI already installed: $(aws --version)"
              fi
              
              # Mount EFS
              echo "Mounting EFS..."
              mkdir -p ${var.efs_mount_point}
              mount -t nfs4 -o nfsvers=4.1,rsize=1048576,wsize=1048576,hard,timeo=600,retrans=2 \
                ${aws_efs_file_system.shared.dns_name}:/ ${var.efs_mount_point}
              
              # Add to fstab for persistence
              echo "${aws_efs_file_system.shared.dns_name}:/ ${var.efs_mount_point} nfs4 nfsvers=4.1,rsize=1048576,wsize=1048576,hard,timeo=600,retrans=2 0 0" >> /etc/fstab
              
              # Create install directory
              echo "Creating install directory..."
              mkdir -p ${var.install_directory}
              cd ${var.install_directory}
              
              # Download setup archive from S3
              echo "Downloading CData Sync archive from S3..."
              aws s3 cp "s3://${aws_s3_bucket.setup_files.id}/${local.setup_script_key}" .
              
              # Extract archive
              echo "Extracting archive..."
              tar -xzf "${var.setup_archive_name}"
              
              # Execute service.sh
              echo "Executing service.sh..."
              if [ -f "service.sh" ]; then
                bash service.sh
              else
                echo "Error: service.sh not found"
                exit 1
              fi
              
              # Create sync directory on EFS after service.sh (cdatasync user should be created)
              echo "Creating sync directory on EFS..."
              mkdir -p ${var.efs_mount_point}/${var.efs_sync_directory}
              
              # Generate properties file
              echo "Generating sync.properties..."
              $(find ${var.install_directory}/jre/ -name java) -jar ${var.install_directory}/sync.jar -GenerateProperties
              
              # Configure RDS connection
              echo "Configuring RDS connection..."
              sed -i "s|^cdata.app.db=.*|cdata.app.db=jdbc:cdata:mysql:server=${aws_db_instance.default_rds.address};port=${aws_db_instance.default_rds.port};database=${var.rds_db_name};user=${var.rds_username};password=${var.rds_password}|" \
                ${var.install_directory}/sync.properties
              
              # Configure EFS directory (point to sync subdirectory)
              echo "Configuring EFS directory..."
              sed -i "s|^cdata.app.directory=.*|cdata.app.directory=${var.efs_mount_point}/${var.efs_sync_directory}|" \
                ${var.install_directory}/sync.properties

              # Change ownership to cdatasync user (created by service.sh)
              echo "Changing ownership of sync directory..."
              if id "cdatasync" &>/dev/null; then
                chown -R cdatasync:cdatasync ${var.efs_mount_point}/${var.efs_sync_directory}
                chmod 755 ${var.efs_mount_point}/${var.efs_sync_directory}
                echo "Ownership changed to cdatasync:cdatasync"
              else
                echo "Error: cdatasync user not found, creating..."
                exit 1
              fi
              
              # Enable and start service
              echo "Enabling and starting cdatasync service..."
              systemctl enable cdatasync
              systemctl start cdatasync
              
              echo "=== CData Sync setup completed successfully ==="
              echo "Timestamp: $(date)"
              EOF
  
  tags = {
    Name = "${var.project_name}-ec2-${count.index + 1}"
  }
  
  depends_on = [
    aws_s3_object.setup_archive,
    aws_efs_mount_target.shared
  ]
}

# Attach all instances to the main target group
resource "aws_lb_target_group_attachment" "app_tg_main_attachment" {
  count            = var.instance_num
  target_group_arn = aws_lb_target_group.app_tg_main.arn
  target_id        = aws_instance.app_ec2[count.index].id
  port             = local.app_port
}

# Attach each instance to its individual target group
resource "aws_lb_target_group_attachment" "app_tg_individual_attachment" {
  count            = var.instance_num
  target_group_arn = aws_lb_target_group.app_tg_individual[count.index].arn
  target_id        = aws_instance.app_ec2[count.index].id
  port             = local.app_port
}

# ------------------------------------------------------------------------------
# Route53 DNS Records
# ------------------------------------------------------------------------------
# Common subdomain (sync.domain) pointing to ALB
resource "aws_route53_record" "app_dns_main" {
  zone_id = data.aws_route53_zone.primary.zone_id
  name    = format("%s.%s", var.common_subdomain, var.domain_name)
  type    = "A"
  alias {
    name                   = aws_lb.app_alb.dns_name
    zone_id                = aws_lb.app_alb.zone_id
    evaluate_target_health = true
  }
}

# Individual subdomains (sync-1.domain, sync-2.domain, etc.) pointing to ALB
resource "aws_route53_record" "app_dns_individual" {
  count   = var.instance_num
  zone_id = data.aws_route53_zone.primary.zone_id
  name    = format("%s.%s", local.instance_subdomains[count.index], var.domain_name)
  type    = "A"
  alias {
    name                   = aws_lb.app_alb.dns_name
    zone_id                = aws_lb.app_alb.zone_id
    evaluate_target_health = true
  }
}

# ------------------------------------------------------------------------------
# RDS Database
# ------------------------------------------------------------------------------
resource "aws_db_subnet_group" "rds_subnet_group" {
  name       = "${var.project_name}-rds-subnet-group"
  subnet_ids = [for subnet in aws_subnet.private : subnet.id]
  tags = {
    Name = "${var.project_name}-rds-subnet-group"
  }
}

resource "aws_db_instance" "default_rds" {
  identifier             = "${var.project_name}-rds-db"
  allocated_storage      = var.rds_allocated_storage
  engine                 = var.rds_engine
  engine_version         = var.rds_engine_version
  instance_class         = var.rds_instance_class
  db_name                = var.rds_db_name
  username               = var.rds_username
  password               = var.rds_password
  db_subnet_group_name   = aws_db_subnet_group.rds_subnet_group.name
  vpc_security_group_ids = [aws_security_group.rds_sg.id]
  skip_final_snapshot    = true
  multi_az               = var.rds_multi_az
  tags = {
    Name = "${var.project_name}-rds-db"
  }
}

# ------------------------------------------------------------------------------
# Variables
# ------------------------------------------------------------------------------

# General Settings
variable "aws_region" {
  description = "AWS region to deploy resources."
  type        = string
  default     = "ap-northeast-1"
}

variable "project_name" {
  description = "A project name to prefix to resource names."
  type        = string
  default     = "cdata-sync"
}

# Network Configuration
variable "vpc_cidr" {
  description = "CIDR block for the VPC. Ensure this does not overlap with 10.0.0.0/16 or 172.31.0.0/16."
  type        = string
}

variable "public_subnet_cidrs" {
  description = "List of CIDR blocks for public subnets."
  type        = list(string)
}

variable "private_subnet_cidrs" {
  description = "List of CIDR blocks for private subnets (for RDS)."
  type        = list(string)
}

# Domain and Certificate Settings
variable "domain_name" {
  description = "The primary domain name for Route 53 (e.g., example.com)."
  type        = string
}

variable "common_subdomain" {
  description = "Common subdomain for load-balanced access (e.g., 'sync' creates sync.example.com)."
  type        = string
  default     = "sync"
}

variable "acm_certificate_arn" {
  description = "ARN of the existing ACM certificate to use for the ALB HTTPS listener."
  type        = string
}

# CData Sync Configuration
variable "instance_num" {
  description = "Number of EC2 instances to create."
  type        = number

  validation {
    condition     = var.instance_num >= 1
    error_message = "The instance_num must be at least 1."
  }
}

variable "ec2_instance_type" {
  description = "Instance type for the EC2 instances."
  type        = string
  default     = "t3.large"
}

variable "ec2_key_name" {
  description = "Name of the EC2 key pair to use for all instances."
  type        = string
}

variable "app_ami_id" {
  description = "AMI ID for the application EC2 instances. If not specified, the latest Ubuntu 24.04 LTS AMI will be used."
  type        = string
  default     = null
}

variable "health_check_path" {
  description = "Path for ALB health checks on EC2 instances."
  type        = string
  default     = "/pub/ping.rst"
}

# Setup Files Configuration
variable "setup_archive_path" {
  description = "Local path to the CData Sync tar.gz file to upload to S3."
  type        = string
}

variable "setup_archive_name" {
  description = "Name of the setup archive file."
  type        = string
}

variable "install_directory" {
  description = "Directory on EC2 where CData Sync will be installed."
  type        = string
  default     = "/opt/sync"
}

variable "efs_mount_point" {
  description = "Mount point for the shared EFS file system."
  type        = string
  default     = "/mnt/efs"
}

variable "efs_sync_directory" {
  description = "Directory name under EFS mount point for CData Sync data."
  type        = string
  default     = "sync"
}

# RDS Database Configuration
variable "rds_instance_class" {
  description = "Instance class for the RDS database."
  type        = string
  default     = "db.t4g.small"
}

variable "rds_allocated_storage" {
  description = "Allocated storage for RDS in GB."
  type        = number
  default     = 20
}

variable "rds_engine" {
  description = "RDS database engine."
  type        = string
  default     = "mysql"
}

variable "rds_engine_version" {
  description = "RDS database engine version."
  type        = string
  default     = "8.4"
}

variable "rds_db_name" {
  description = "Name of the initial database in RDS."
  type        = string
  default     = "cdatasync"
}

variable "rds_username" {
  description = "Master username for the RDS database."
  type        = string
  default     = "adminuser"
}

variable "rds_password" {
  description = "Master password for the RDS database."
  type        = string
  sensitive   = true
}

variable "rds_multi_az" {
  description = "Specifies if the RDS instance is multi-AZ."
  type        = bool
  default     = false
}

# ------------------------------------------------------------------------------
# Outputs
# ------------------------------------------------------------------------------

output "vpc_id" {
  description = "ID of the created VPC."
  value       = aws_vpc.main.id
}

output "setup_s3_bucket" {
  description = "S3 bucket name for setup files."
  value       = aws_s3_bucket.setup_files.id
}

output "efs_id" {
  description = "EFS file system ID."
  value       = aws_efs_file_system.shared.id
}

output "efs_dns_name" {
  description = "EFS DNS name for mounting."
  value       = aws_efs_file_system.shared.dns_name
}

output "ami_id_used" {
  description = "AMI ID used for EC2 instances."
  value       = var.app_ami_id != null ? var.app_ami_id : data.aws_ami.ubuntu.id
}

output "ami_description" {
  description = "Description of the AMI used."
  value       = var.app_ami_id != null ? "Custom AMI" : data.aws_ami.ubuntu.description
}

output "public_subnet_ids" {
  description = "IDs of the public subnets."
  value       = [for subnet in aws_subnet.public : subnet.id]
}

output "private_subnet_ids_for_rds" {
  description = "IDs of the private subnets used for RDS."
  value       = [for subnet in aws_subnet.private : subnet.id]
}

output "alb_dns_name" {
  description = "DNS name for the Application Load Balancer."
  value       = aws_lb.app_alb.dns_name
}

output "app_ec2_instance_ids" {
  description = "EC2 instance IDs for the application stacks."
  value = {
    for i in range(var.instance_num) :
    local.instance_subdomains[i] => aws_instance.app_ec2[i].id
  }
}

output "ec2_public_ips" {
  description = "Public IP addresses of EC2 instances (for direct SSH access if needed)."
  value = {
    for i in range(var.instance_num) :
    local.instance_subdomains[i] => aws_instance.app_ec2[i].public_ip
  }
}

output "ec2_private_ips" {
  description = "Private IP addresses of EC2 instances."
  value = {
    for i in range(var.instance_num) :
    local.instance_subdomains[i] => aws_instance.app_ec2[i].private_ip
  }
}

output "app_main_url" {
  description = "Main application URL (load balanced across all instances)."
  value       = format("https://%s.%s", var.common_subdomain, var.domain_name)
}

output "app_individual_urls" {
  description = "Individual instance URLs."
  value       = [for subdomain in local.instance_subdomains : format("https://%s.%s", subdomain, var.domain_name)]
}

output "rds_endpoint" {
  description = "Endpoint address for the RDS database instance."
  value       = aws_db_instance.default_rds.endpoint
}
